# 機能設計書 22-Security CSRF

## 概要

本ドキュメントは、Symfony Security CSRFコンポーネントの機能設計を記述する。このコンポーネントは、CSRF（クロスサイトリクエストフォージェリ）保護ライブラリを提供し、CSRFトークンの生成・検証機能を実現する。

### 本機能の処理概要

**業務上の目的・背景**：CSRF攻撃は、ユーザーが意図しないリクエストを悪意のあるサイトから送信させる攻撃手法である。Webアプリケーションにおいてフォーム送信や状態変更を伴うリクエストに対して、CSRFトークンによる正当性検証を行うことで、この攻撃を防止する必要がある。本コンポーネントはトークンの生成、保存、検証をセキュアに実装する。

**機能の利用シーン**：フォーム送信時のCSRFトークン検証、ログアウトリクエストの正当性確認、API呼び出しにおけるリクエスト発信元の検証、Same-Origin検証によるステートレスCSRF保護など、状態変更を伴うHTTPリクエスト全般で使用される。

**主要な処理内容**：
1. CsrfTokenManagerによるトークンID付きCSRFトークンの生成（初回生成時にトークンストレージに保存）
2. トークン値のランダム化（XOR + ランダムキーによるマスキング）によるBREACH攻撃対策
3. isTokenValidによるトークン検証（デランダム化後のhash_equalsによるタイミングセーフ比較）
4. SameOriginCsrfTokenManagerによるステートレスCSRF保護（Sec-Fetch-Site、Origin/Referer、ダブルサブミット方式）
5. HTTPSプレフィクス付きCookieによるHTTPチャネル経由の偽造防止
6. セッションベースの検証方式ダウングレード防止

**関連システム・外部連携**：Security HTTPコンポーネント（CsrfProtectionListener）、Formコンポーネント（フォームCSRF拡張）、HttpFoundation（セッション、Cookie、リクエストヘッダー）と連携する。

**権限による制御**：CSRFトークンの検証はリクエストの正当性確認であり、ロールベースの権限制御とは独立して動作する。全ユーザーに対して均一に適用される。

## 関連画面

| 画面No | 画面名 | 関連種別 | 関連する操作・処理 |
|--------|--------|----------|------------------|
| - | （直接的な画面関連なし） | - | フォーム送信時の隠しフィールドとしてCSRFトークンが埋め込まれる |

## 機能種別

セキュリティ / トークン生成・検証

## 入力仕様

### 入力パラメータ

| パラメータ名 | 型 | 必須 | 説明 | バリデーション |
|-------------|-----|-----|------|---------------|
| tokenId | string | Yes | トークンの識別子（用途ごとに異なる） | 空文字列不可 |
| token | CsrfToken | Yes（検証時） | 検証対象のCSRFトークン | CsrfTokenインスタンス |
| namespace | string/RequestStack/callable/null | No | トークンの名前空間（HTTPS識別用） | string、RequestStack、callableまたはnull |

### 入力データソース

フォーム隠しフィールド、リクエストパラメータ、リクエストヘッダー、Cookie、セッション

## 出力仕様

### 出力データ

| 項目名 | 型 | 説明 |
|--------|-----|------|
| CsrfToken | CsrfToken | 生成されたCSRFトークン（ID + ランダム化された値） |
| isValid | bool | トークン検証結果 |
| removedToken | string/null | 削除されたトークンの値（removeToken時） |

### 出力先

セッション（トークンストレージ経由）、フォームの隠しフィールド、HTTPレスポンスCookie

## 処理フロー

### 処理シーケンス

```
1. トークン生成（getToken）
   └─ ストレージにトークンが存在すれば取得、なければ新規生成して保存
2. トークンランダム化
   └─ 32バイトのランダムキーでXOR暗号化し、Base64エンコードして返却
3. トークン検証（isTokenValid）
   └─ 提出されたトークンをデランダム化し、ストレージの値とhash_equalsで比較
4. SameOrigin検証（SameOriginCsrfTokenManager）
   └─ Sec-Fetch-Site、Origin/Referer、ダブルサブミット方式で検証
```

### フローチャート

```mermaid
flowchart TD
    A[CSRFトークン要求] --> B{ストレージにトークンあり?}
    B -->|Yes| C[ストレージから取得]
    B -->|No| D[新規トークン生成]
    D --> E[ストレージに保存]
    C --> F[トークン値ランダム化]
    E --> F
    F --> G[CsrfTokenオブジェクト返却]

    H[CSRFトークン検証要求] --> I[トークン値デランダム化]
    I --> J{ストレージにトークンあり?}
    J -->|No| K[false返却]
    J -->|Yes| L[hash_equals比較]
    L --> M{一致?}
    M -->|Yes| N[true返却]
    M -->|No| K
```

## ビジネスルール

### 業務ルール

| ルールNo | ルール名 | 内容 | 適用条件 |
|---------|---------|------|---------|
| BR-22-01 | トークンランダム化 | 毎回異なるランダムキーでXOR暗号化し、BREACH攻撃を防止する | トークン生成・返却時 |
| BR-22-02 | 名前空間分離 | HTTPS/HTTP環境で異なる名前空間を使用し、プロトコル間のトークン漏洩を防止 | トークン操作時 |
| BR-22-03 | タイミングセーフ比較 | hash_equalsを使用し、タイミング攻撃によるトークン推測を防止 | トークン検証時 |
| BR-22-04 | SameOriginダウングレード防止 | セッションに以前の検証方式を記録し、より安全な方式からの変更を拒否 | SameOriginCsrfTokenManager使用時 |
| BR-22-05 | Cookie __Host-プレフィクス | HTTPS接続時はCookieに__Host-プレフィクスを付与し、HTTPチャネル経由の偽造を防止 | HTTPS環境でのダブルサブミット時 |

### 計算ロジック

トークンランダム化: `randomize(value) = xxh128(key)[0..hash_len] + "." + base64url(key) + "." + base64url(value XOR key)`（keyは32バイトのランダム値）

## データベース操作仕様

### 操作別データベース影響一覧

直接的なデータベース操作は行わない。トークンはTokenStorageInterface経由で保存され、デフォルトではNativeSessionTokenStorage（PHPセッション）を使用する。

## エラー処理

### エラーケース一覧

| エラーコード | エラー種別 | 発生条件 | 対処方法 |
|------------|----------|---------|---------|
| - | InvalidArgumentException | namespace引数が不正な型 | 正しい型（string、RequestStack、callable、null）を指定 |
| - | InvalidArgumentException | SameOriginCsrfTokenManagerでCookie名が空または不正な文字を含む | 有効なCookie名を指定 |
| - | LogicException | SameOriginCsrfTokenManagerでリクエストコンテキストがない | リクエストスタックにリクエストが存在することを確認 |

### リトライ仕様

リトライは不要。トークン検証失敗時はフォーム再表示や403エラーで対応する。

## トランザクション仕様

直接的なトランザクション管理は行わない。セッション操作はPHPのセッション機構に依存する。

## パフォーマンス要件

トークン生成・検証はリクエスト処理中に行われるため、低レイテンシーであることが期待される。random_bytes(32)による暗号学的に安全な乱数生成、XOR演算、hash_equalsは全て高速に動作する。

## セキュリティ考慮事項

- BREACH攻撃対策：毎回異なるランダムキーでトークン値をXOR暗号化し、レスポンス圧縮経由のトークン漏洩を防止
- タイミング攻撃対策：hash_equalsによる固定時間比較
- HTTPSプロトコル分離：名前空間でHTTPS/HTTPのトークンを分離
- SensitiveParameterアトリビュート：CsrfToken値がスタックトレースに露出しないよう保護
- __Host-Cookie プレフィクス：HTTPS接続時にCookieの偽造を防止
- ダウングレード防止：セッション記録により検証方式の変更を検出・拒否

## 備考

Symfony 8.1では、SameOriginCsrfTokenManagerが推奨されるステートレスCSRF保護方式として提供されている。従来のセッションベースのCsrfTokenManagerも引き続き利用可能。SameOriginCsrfListenerへの移行が推奨されている（clearCookies、persistStrategy、onKernelResponseメソッドは非推奨）。

---

## コードリーディングガイド

本機能を理解するために参照すべきファイルと、推奨する読み解き順序を以下に示す。

### 推奨読解順序

#### Step 1: データ構造を理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 1-1 | CsrfToken.php | `src/Symfony/Component/Security/Csrf/CsrfToken.php` | CSRFトークンのデータ構造（id + value）を理解。valueには#[SensitiveParameter]属性が付与されている |
| 1-2 | CsrfTokenManagerInterface.php | `src/Symfony/Component/Security/Csrf/CsrfTokenManagerInterface.php` | トークンマネージャーの4メソッド（getToken, refreshToken, removeToken, isTokenValid）を理解 |
| 1-3 | TokenStorageInterface.php | `src/Symfony/Component/Security/Csrf/TokenStorage/TokenStorageInterface.php` | トークンストレージの契約を理解 |
| 1-4 | TokenGeneratorInterface.php | `src/Symfony/Component/Security/Csrf/TokenGenerator/TokenGeneratorInterface.php` | トークン生成器のインターフェース |

**読解のコツ**: CsrfTokenは不変オブジェクトとして設計されており、idとvalueの組み合わせで一意に識別される。__toString()がvalueを返すため、テンプレート内で直接文字列として使用可能。

#### Step 2: 従来型トークンマネージャーを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 2-1 | CsrfTokenManager.php | `src/Symfony/Component/Security/Csrf/CsrfTokenManager.php` | セッションベースのCSRFトークンマネージャーの中核実装 |

**主要処理フロー**:
1. **40-64行目**: コンストラクタでgenerator, storage, namespaceを初期化。namespaceはHTTPS判定用
2. **66-78行目**: getTokenでストレージ確認 -> 存在すれば取得、なければ生成して保存 -> randomize()で暗号化して返却
3. **95-103行目**: isTokenValidでデランダム化後にhash_equalsで比較
4. **110-116行目**: randomize()で32バイトランダムキーによるXOR暗号化 + Base64エンコード
5. **118-131行目**: derandomize()でBase64デコード + XOR復号

#### Step 3: SameOrigin型トークンマネージャーを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 3-1 | SameOriginCsrfTokenManager.php | `src/Symfony/Component/Security/Csrf/SameOriginCsrfTokenManager.php` | ステートレスCSRF保護の実装 |

**主要処理フロー**:
- **89-96行目**: getTokenでCookie名をトークン値として返却（ステートレス）
- **116-191行目**: isTokenValidでSec-Fetch-Site -> Origin/Referer -> ダブルサブミットの順で検証
- **251-272行目**: isValidOriginでSec-Fetch-SiteまたはOrigin/Refererヘッダーを検証
- **277-306行目**: isValidDoubleSubmitでCookie + ヘッダーのダブルサブミットを検証
- **150-170行目**: セッションによる検証方式ダウングレード防止

#### Step 4: トークンストレージを理解する

| 順序 | ファイル | パス | 読解ポイント |
|-----|---------|------|-------------|
| 4-1 | NativeSessionTokenStorage.php | `src/Symfony/Component/Security/Csrf/TokenStorage/NativeSessionTokenStorage.php` | PHPネイティブセッションベースのストレージ |
| 4-2 | SessionTokenStorage.php | `src/Symfony/Component/Security/Csrf/TokenStorage/SessionTokenStorage.php` | Symfonyセッションベースのストレージ |

### プログラム呼び出し階層図

```
CsrfTokenManager::getToken()
    |
    +-- getNamespace()                    [名前空間取得]
    +-- TokenStorage::hasToken()          [トークン存在確認]
    +-- TokenStorage::getToken()          [トークン取得]
    +-- TokenGenerator::generateToken()   [新規トークン生成]
    +-- TokenStorage::setToken()          [トークン保存]
    +-- randomize()                       [XOR暗号化]
         +-- random_bytes(32)
         +-- xor()
         +-- base64_encode()

CsrfTokenManager::isTokenValid()
    |
    +-- getNamespace()
    +-- TokenStorage::hasToken()
    +-- TokenStorage::getToken()
    +-- derandomize()                     [XOR復号]
         +-- base64_decode()
         +-- xor()
    +-- hash_equals()                     [タイミングセーフ比較]

SameOriginCsrfTokenManager::isTokenValid()
    |
    +-- isValidOrigin()                   [Origin検証]
    |    +-- Sec-Fetch-Site検査
    |    +-- Origin/Refererヘッダー検査
    |
    +-- isValidDoubleSubmit()             [ダブルサブミット検証]
    |    +-- Cookie値検査
    |    +-- ヘッダー値検査
    |
    +-- Session検証方式ダウングレード防止
```

### データフロー図

```
[入力]                       [処理]                          [出力]

tokenId -----------------> CsrfTokenManager::getToken() --> CsrfToken(id, randomized_value)
                                |
TokenStorage <-----------> hasToken / getToken / setToken
                                |
TokenGenerator ----------> generateToken()

CsrfToken(id, value) ---> isTokenValid() ----------------> bool
                                |
                           derandomize(value)
                                |
                           hash_equals(stored, derandomized)

Request -----------------> SameOriginCsrfTokenManager ----> bool
  (Headers, Cookies)       ::isTokenValid()
```

### 関連ファイル一覧

| ファイル | パス | 種別 | 役割 |
|---------|------|------|------|
| CsrfToken.php | `src/Symfony/Component/Security/Csrf/CsrfToken.php` | ソース | CSRFトークンデータクラス |
| CsrfTokenManager.php | `src/Symfony/Component/Security/Csrf/CsrfTokenManager.php` | ソース | セッションベーストークンマネージャー |
| CsrfTokenManagerInterface.php | `src/Symfony/Component/Security/Csrf/CsrfTokenManagerInterface.php` | ソース | トークンマネージャーインターフェース |
| SameOriginCsrfTokenManager.php | `src/Symfony/Component/Security/Csrf/SameOriginCsrfTokenManager.php` | ソース | ステートレスCSRFトークンマネージャー |
| SameOriginCsrfListener.php | `src/Symfony/Component/Security/Csrf/SameOriginCsrfListener.php` | ソース | SameOriginレスポンスリスナー |
| TokenGeneratorInterface.php | `src/Symfony/Component/Security/Csrf/TokenGenerator/TokenGeneratorInterface.php` | ソース | トークン生成器インターフェース |
| UriSafeTokenGenerator.php | `src/Symfony/Component/Security/Csrf/TokenGenerator/UriSafeTokenGenerator.php` | ソース | URI安全トークン生成器 |
| TokenStorageInterface.php | `src/Symfony/Component/Security/Csrf/TokenStorage/TokenStorageInterface.php` | ソース | トークンストレージインターフェース |
| NativeSessionTokenStorage.php | `src/Symfony/Component/Security/Csrf/TokenStorage/NativeSessionTokenStorage.php` | ソース | ネイティブセッションストレージ |
| SessionTokenStorage.php | `src/Symfony/Component/Security/Csrf/TokenStorage/SessionTokenStorage.php` | ソース | Symfonyセッションストレージ |
